package ch.frankel.blog.lombok.handler;

import static lombok.AccessLevel.PACKAGE;
import static lombok.AccessLevel.PRIVATE;
import static lombok.AccessLevel.PROTECTED;
import static lombok.AccessLevel.PUBLIC;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import lombok.core.AnnotationValues;
import lombok.javac.JavacAnnotationHandler;
import lombok.javac.JavacNode;
import lombok.javac.handlers.JavacHandlerUtil;
import lombok.javac.handlers.JavacHandlerUtil.MemberExistsResult;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.frankel.blog.lombok.Delegate;

import com.sun.tools.javac.code.Symbol;
import com.sun.tools.javac.code.Type;
import com.sun.tools.javac.code.Symbol.ClassSymbol;
import com.sun.tools.javac.tree.JCTree;
import com.sun.tools.javac.tree.TreeMaker;
import com.sun.tools.javac.tree.JCTree.JCAnnotation;
import com.sun.tools.javac.tree.JCTree.JCBlock;
import com.sun.tools.javac.tree.JCTree.JCClassDecl;
import com.sun.tools.javac.tree.JCTree.JCExpression;
import com.sun.tools.javac.tree.JCTree.JCIdent;
import com.sun.tools.javac.tree.JCTree.JCMethodDecl;
import com.sun.tools.javac.tree.JCTree.JCMethodInvocation;
import com.sun.tools.javac.tree.JCTree.JCModifiers;
import com.sun.tools.javac.tree.JCTree.JCStatement;
import com.sun.tools.javac.tree.JCTree.JCTypeApply;
import com.sun.tools.javac.tree.JCTree.JCTypeParameter;
import com.sun.tools.javac.tree.JCTree.JCVariableDecl;
import com.sun.tools.javac.util.List;
import com.sun.tools.javac.util.Name;

/**
 * Handles {@link Delegate} annotations.
 * 
 * @author <a href="http://blog.frankel.ch/">Nicolas Frankel</a>
 */
public class JDelegateHandler implements JavacAnnotationHandler<Delegate> {

    /** Logger instance. */
    private static final Logger LOGGER = LoggerFactory.getLogger(JDelegateHandler.class);

    /** Parameter's types mapper. */
    private Mapper<Class<?>, Type> paramMapper;

    /** Return types mapper. */
    private final Mapper<Class<?>, Integer> returnMapper;

    /**
     * Constructor.
     */
    public JDelegateHandler() {

        LOGGER.info("Handler referenced {}", getClass());

        returnMapper = new ReturnTypeMapper();
    }

    /**
     * @see lombok.javac.JavacAnnotationHandler#handle(lombok.core.AnnotationValues,
     *      com.sun.tools.javac.tree.JCTree.JCAnnotation,
     *      lombok.javac.JavacNode)
     */
    @Override
    public boolean handle(AnnotationValues<Delegate> annotationValues, JCAnnotation annotation, JavacNode annotationNode) {

        paramMapper = new ParamTypeMapper(annotationNode);

        JavacNode fieldNode = annotationNode.up();

        Name delegateName = getAttributeClassName(fieldNode);

        JavacNode typeNode = fieldNode.up();

        Name delegatorName = getTypeClassName(typeNode);

        try {

            Class<?> delegateClass = Class.forName(delegateName.toString());

            injectDelegatesMethod(delegatorName, delegateClass, fieldNode);

        } catch (ClassNotFoundException e) {

            annotationNode.addError("Impossible to load " + e.getMessage());
        }

        return true;
    }

    /**
     * Inject the delegates method in the class body.
     * 
     * @param delegatorName
     * @param delegateClass
     *            Delegated class
     * @param fieldNode
     *            Field node
     */
    private void injectDelegatesMethod(Name delegatorName, Class<?> delegateClass, JavacNode fieldNode) {

        Method methods[] = delegateClass.getDeclaredMethods();

        for (Method method : methods) {

            if (isEligible(method, fieldNode)) {

                JavacHandlerUtil.injectMethod(fieldNode.up(), createDeclarationFromMethod(method, fieldNode));
            }
        }
    }

    /**
     * Checks whether a method is eligible for delegation.
     * 
     * @param method
     *            Method to copy
     * @param fieldNode
     *            Node
     * @return Whether this method should be copied or not
     */
    private boolean isEligible(Method method, JavacNode fieldNode) {

        int modifiers = method.getModifiers();

        if (!Modifier.isPublic(modifiers)) {

            return false;
        }

        MemberExistsResult exists = JavacHandlerUtil.methodExists(method.getName(), fieldNode);

        if (exists == MemberExistsResult.EXISTS_BY_USER) {

            fieldNode.addWarning("An existing method exist with the same name '" + method.getName()
                    + "'. Method will not be delegated.");
        }

        return exists != MemberExistsResult.EXISTS_BY_USER;
    }

    /**
     * Creates the method declaration.
     * 
     * @param method
     *            Method
     * @param fieldNode
     *            Node
     * @return Method declaration
     */
    private JCMethodDecl createDeclarationFromMethod(Method method, JavacNode fieldNode) {

        TreeMaker maker = fieldNode.getTreeMaker();

        String objectName = getObjectName(fieldNode).toString();

        // myObject.myMethod()
        JCExpression callExpr = JavacHandlerUtil.chainDots(maker, fieldNode, objectName, method.getName());

        List<JCVariableDecl> parameters = createArgumentsFromMethod(method, fieldNode);

        JCMethodInvocation delegateInvocation = maker.Apply(List.<JCExpression> nil(), callExpr, maker
                .Idents(parameters));

        JCStatement statement;

        if (method.getReturnType() == void.class) {

            // return myObject.myMethod()
            statement = maker.Exec(delegateInvocation);

        } else {

            // myObject.myMethod()
            statement = maker.Return(delegateInvocation);
        }

        JCBlock methodBody = maker.Block(0, List.of(statement));

        List<JCTypeParameter> methodGenericParams = List.nil();

        List<JCExpression> throwsClauses = createThrowsFromMethod(method, fieldNode);

        long access = convertModifierToAccess(method.getModifiers());

        JCExpression returnExpr = createReturnTypeFromMethod(method, fieldNode);

        JCModifiers modifiers = maker.Modifiers(access);

        Name name = fieldNode.toName(method.getName());

        return maker.MethodDef(modifiers, name, returnExpr, methodGenericParams, parameters, throwsClauses, methodBody,
                null);
    }

    /**
     * Bridge method between access modifier of the Reflection API to Sun's
     * compiler.
     * 
     * @param bits
     *            Modifier bits as returned by {@link Field#getModifiers()}
     * @return long
     */
    private long convertModifierToAccess(int bits) {

        long access = JavacHandlerUtil.toJavacModifier(PACKAGE);

        if (Modifier.isPublic(bits)) {

            access = JavacHandlerUtil.toJavacModifier(PUBLIC);

        } else if (Modifier.isProtected(bits)) {

            access = JavacHandlerUtil.toJavacModifier(PROTECTED);

        } else if (Modifier.isPrivate(bits)) {

            access = JavacHandlerUtil.toJavacModifier(PRIVATE);
        }
        return access;
    }

    /**
     * Compute the attribute class's {@link Name}.
     * 
     * @param fieldNode
     *            Field node
     * @return Class name
     */
    private Name getAttributeClassName(JavacNode fieldNode) {

        JCVariableDecl fieldDecl = (JCVariableDecl) fieldNode.get();

        JCTree type = fieldDecl.getType();

        ClassSymbol symbol = null;

        // Generics
        if (type instanceof JCTypeApply) {

            JCTypeApply apply = (JCTypeApply) type;

            symbol = (ClassSymbol) apply.clazz.type.tsym;

        } else {

            // No generics
            JCIdent varType = (JCIdent) type;

            symbol = (ClassSymbol) varType.sym;
        }

        return symbol.fullname;
    }

    /**
     * Compute the type class's {@link Name}.
     * 
     * @param typeNode
     *            Type node
     * @return Class name
     */
    private Name getTypeClassName(JavacNode typeNode) {

        JCClassDecl classDecl = (JCClassDecl) typeNode.get();

        return classDecl.sym.fullname;
    }

    /**
     * Compute the object's {@link Name}.
     * 
     * @param fieldNode
     *            Field node
     * @return Field name
     */
    private Name getObjectName(JavacNode fieldNode) {

        JCVariableDecl fieldDecl = (JCVariableDecl) fieldNode.get();

        Symbol symbol = fieldDecl.sym;

        return symbol.name;
    }

    /**
     * Create the return type's expression.
     * 
     * @param method
     *            Method
     * @param fieldNode
     *            Attribute node
     * @return Matching expressions
     */
    private JCExpression createReturnTypeFromMethod(Method method, JavacNode fieldNode) {

        TreeMaker maker = fieldNode.getTreeMaker();

        Class<?> clazz = method.getReturnType();

        JCExpression returnExpr = null;

        Integer type = returnMapper.get(clazz);

        if (type == null) {

            returnExpr = JavacHandlerUtil.chainDots(maker, fieldNode, clazz.getName().split("\\."));

        } else {

            returnExpr = maker.TypeIdent(type);
        }

        return returnExpr;
    }

    /**
     * Creates the argument's list
     * 
     * @param method
     *            Method
     * @param fieldNode
     *            Attribute node
     * @return Matching attributes list
     */
    private List<JCVariableDecl> createArgumentsFromMethod(Method method, JavacNode fieldNode) {

        TreeMaker maker = fieldNode.getTreeMaker();

        Class<?>[] classes = method.getParameterTypes();

        JCVariableDecl[] types = new JCVariableDecl[classes.length];

        for (int i = 0; i < classes.length; i++) {

            classes[i].getTypeParameters();

            Name name = fieldNode.toName(classes[i].getName() + i);

            JCVariableDecl varDecl = maker.Param(name, paramMapper.get(classes[i]), null);

            types[i] = varDecl;
        }

        return List.from(types);
    }

    /**
     * Creates the <code>throws</code> clause list
     * 
     * @param method
     *            Method
     * @param fieldNode
     *            Attribute node
     * @return Matching throws clause list
     */
    private List<JCExpression> createThrowsFromMethod(Method method, JavacNode fieldNode) {

        TreeMaker maker = fieldNode.getTreeMaker();

        Class<?>[] classes = method.getExceptionTypes();

        JCExpression[] types = new JCExpression[classes.length];

        for (int i = 0; i < classes.length; i++) {

            types[i] = JavacHandlerUtil.chainDots(maker, fieldNode, classes[i].getName().split("\\."));
        }

        return List.from(types);
    }
}
